document.addEventListener("DOMContentLoaded", async () => {
  const container = document.getElementById("table-list");

  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

  if (!tab || !tab.id || !tab.url) {
    container.innerText = "No active tab found.";
    return;
  }

  const isToolsClub = tab.url.includes("toolsclub.net") || tab.url.includes("semrush");

  try {
    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      files: ["content.js"],
    });
  } catch (e) {
    container.innerText = "Can't run on this page (Chrome pages / Web Store pages are blocked).";
    return;
  }

  chrome.tabs.sendMessage(tab.id, { action: "getTables" }, (response) => {
    if (!response || !response.tables || !response.tables.length) {
      container.innerText = "No tables found on this page.";
      return;
    }

    container.innerHTML = "";

    response.tables.forEach((table, tIndex) => {
      const div = document.createElement("div");
      div.className = "table-container";

      const tableTitle = table.name ? table.name : `Table ${tIndex + 1}`;
      div.innerHTML = `<h2>${escapeHtml(tableTitle)}</h2>`;

      const rows = table.rows || [];
      const colCount = Math.max(...rows.map((r) => r.length), 0);

      // ===== Preview table with copy icons =====
      const tableEl = document.createElement("table");
      tableEl.style.borderCollapse = "collapse";
      tableEl.style.width = "100%";

      const thead = tableEl.createTHead();
      const headRow = thead.insertRow();

      for (let c = 0; c < colCount; c++) {
        const th = document.createElement("th");

        const img = document.createElement("img");
        img.src = "copy.jpg";
        img.style.width = "18px";
        img.style.cursor = "pointer";
        img.title = `Copy column ${c + 1}`;

        img.onclick = () => {
          const colData = rows.map((r) => r[c] || "").join("\n");
          navigator.clipboard.writeText(colData);
        };

        th.appendChild(img);
        headRow.appendChild(th);
      }

      const tbody = tableEl.createTBody();
      rows.forEach((r) => {
        const tr = tbody.insertRow();
        for (let c = 0; c < colCount; c++) {
          const td = tr.insertCell();
          td.innerText = r[c] || "";
          td.style.border = "1px solid #ccc";
          td.style.padding = "4px";
        }
      });

      div.appendChild(tableEl);

      // ===== Export CSV =====
      const btnCsv = document.createElement("button");
      btnCsv.innerText = "export CSV";
      btnCsv.style.marginTop = "10px";
      btnCsv.style.padding = "6px 10px";
      btnCsv.style.cursor = "pointer";

      btnCsv.addEventListener("click", () => {
        const filename = isToolsClub ? "SEMrush_table.csv" : `table_${tIndex + 1}.csv`;
        exportTableToCSV(rows, filename, isToolsClub);
      });

      div.appendChild(btnCsv);

      // ===== Export REAL XLS (BIFF8) =====
      const btnXls = document.createElement("button");
      btnXls.innerText = "export XLS";
      btnXls.style.marginTop = "10px";
      btnXls.style.marginLeft = "10px";
      btnXls.style.padding = "6px 10px";
      btnXls.style.cursor = "pointer";
      btnXls.style.backgroundColor = "#2e7d32";
      btnXls.style.color = "white";
      btnXls.style.border = "none";
      btnXls.style.borderRadius = "6px";

      btnXls.addEventListener("click", () => {
        const filename = isToolsClub ? "SEMrush_table.xls" : `table_${tIndex + 1}.xls`;
        exportTableToRealXLS(rows, filename, isToolsClub, tableTitle);
      });

      div.appendChild(btnXls);

      container.appendChild(div);
    });
  });
});

// =========================
// CSV export (semicolon ;)
// =========================
function exportTableToCSV(rows, filename, isToolsClub) {
  if (!filename.toLowerCase().endsWith(".csv")) filename += ".csv";

  const allRows = buildAllRows(rows, isToolsClub);

  const csvBody = allRows
    .map((row) =>
      row
        .map((v) => {
          const val = v == null ? "" : v.toString();
          return /[;"\n\r]/.test(val) ? `"${val.replace(/"/g, '""')}"` : val;
        })
        .join(";")
    )
    .join("\r\n");

  const csvContent = "\uFEFF" + "sep=;\r\n" + csvBody;
  const blob = new Blob([csvContent], { type: "text/csv;charset=utf-8" });
  downloadBlob(blob, filename);
}

// ==========================================
// REAL XLS (BIFF8) export using SheetJS
// ==========================================
function exportTableToRealXLS(rows, filename, isToolsClub, sheetName = "Table") {
  if (!filename.toLowerCase().endsWith(".xls")) filename += ".xls";

  // SheetJS must exist
  if (!window.XLSX) {
    alert("SheetJS (XLSX) library not loaded. Include xlsx.full.min.js in popup.html before popup.js");
    return;
  }

  const allRows = buildAllRows(rows, isToolsClub);

  // Convert numeric-looking strings to numbers (optional)
  const normalized = allRows.map((row, rIdx) =>
    row.map((cell) => {
      if (rIdx === 0) return (cell ?? "").toString(); // headers as strings
      const s = (cell ?? "").toString().trim();
      const raw = s.replace(/,/g, "");
      const num = Number(raw);
      return s !== "" && Number.isFinite(num) && /^-?\d+(\.\d+)?$/.test(raw) ? num : (cell ?? "").toString();
    })
  );

  const wb = XLSX.utils.book_new();
  const ws = XLSX.utils.aoa_to_sheet(normalized);

  const safeSheet = (sheetName || "Table").toString().slice(0, 31);
  XLSX.utils.book_append_sheet(wb, ws, safeSheet);

  // ✅ TRUE .xls (BIFF8) -> Excel opens without “extension mismatch” warning
  XLSX.writeFile(wb, filename, { bookType: "biff8" });
}

// ========= helpers =========
function buildAllRows(rows, isToolsClub) {
  if (isToolsClub) {
    const headers = ["Domain", "Authority", "Visits"];
    const data = rows.map((row) => {
      const target = (row[0] || "").split("\n")[0].trim();
      let authority = (row[1] || "0").toString();
      let visits = (row[4] || "0").toString();
      if (authority.toLowerCase() === "n/a") authority = "0";
      if (visits.toLowerCase() === "n/a") visits = "0";
      return [target, authority, visits];
    });
    return [headers, ...data];
  }

  const maxCols = Math.max(...rows.map((r) => r.length), 0);
  const headers = Array.from({ length: maxCols }, (_, i) => `Column ${i + 1}`);
  return [headers, ...rows.map((r) => r.map((v) => (v ?? "").toString()))];
}

function downloadBlob(blob, filename) {
  const url = URL.createObjectURL(blob);
  chrome.downloads.download(
    { url, filename, saveAs: true },
    () => setTimeout(() => URL.revokeObjectURL(url), 15000)
  );
}

function escapeHtml(str) {
  return (str ?? "")
    .toString()
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#039;");
}
